
/* VARIABLES USED IN REGRESSIONS CODE:
lpo: LoanValues 
ptpno: Participation
bnpl: Borrower CPRisk
bsfln: Borrower Collateral
bcgln: Borrower AGS
bsgln: Borrower Semis
bis: LB, i.e. indicator for post-Lehman Brothers default
lbcpos: binary variable indicating borrower-lender-collateral unit had positive loans outstanding in sample
br: factor variable for borrowers
ln: factor variable for lenders
day: factor variable for date
cg: indicator for if observation from AGS repo market 
sg: indicator for if observation from semis repo market
td: TED spread lagged one day
UNDERSCORES BETWEEN VARIABLE NAMES REPRESENT INTERACTIONS.
*/

clear all

ssc install reghdfe
ssc install tuples
ssc install estout

cd "[FOLDER PATH]"
import delimited using "[DATA FILE]"

set more off

/* Convert variables from string to integer */
encode lnd, generate(ln)
encode brw, generate(br)
encode iss, generate(clt)
rename tedlg ted

rename ted td
gen const = day > 0

drop if day > 39724

gen bis = day > 39705
order bis, after(td)

egen new_day = tag(day)
egen tedmean = mean(td) if new_day == 1
replace tedmean = tedmean[_n-1] if tedmean == .
egen tedsd = sd(td) if new_day == 1
replace tedsd = tedsd[_n-1] if tedsd == .
replace td = (td - tedmean) / tedsd


// Remove NAs and replace with missing value, first force variables to string to 
//   make it easier.
foreach v of varlist spr-bcgs {
	tostring `v', force replace
	replace `v' = "." if `v' == "NA" | `v' == "-Inf" | `v' == "Inf"
}
destring, replace

generate cg = iss == "cgs"
generate sg = iss == "sgs"
generate fp = iss == "frn" | iss == "prv"
generate ot = sg + fp
generate no = iss == "none"

replace bcgs = bcgs / 1000000000
replace bsgs = bsgs / 1000000000
replace lcgs = lcgs / 1000000000
replace lsgs = lsgs / 1000000000

/* geenrate and format the sec holdings variables */
generate lsfln = ln(lcgs + lsg + 1)
generate lsgln = ln(lsgs + 1)
generate lcgln = ln(lcgs + 1)
generate bsfln = ln(bcgs + bsgs + 1)
generate bsgln = ln(bsgs + 1)
generate bcgln = ln(bcgs + 1)
generate lsf = lcgs + lsgs
generate bsf = bcgs + bsgs

// Standardise the main variables
egen new_brw = tag(brw)
egen new_lnd = tag(lnd)
foreach v of varlist bnpl bsfln bsgln bcgln {
	egen `v'mn = mean(`v') if new_brw == 1
	replace `v'mn = `v'mn[_n-1] if `v'mn == .
	egen `v'sd = sd(`v') if new_brw == 1
	replace `v'sd = `v'sd[_n-1] if `v'sd == .
	replace `v' = (`v' - `v'mn) / `v'sd
}

// Now the lender variables
foreach v of varlist lnpl lsfln lsgln lcgln {
	egen `v'mn = mean(`v') if new_lnd == 1
	replace `v'mn = `v'mn[_n-1] if `v'mn == .
	egen `v'sd = sd(`v') if new_brw == 1
	replace `v'sd = `v'sd[_n-1] if `v'sd == .
	replace `v' = (`v' - `v'mn) / `v'sd

}

order lsfln-lcgln, after(lcgs)
order bsfln-bcgln, after(bcgs)

/* different DVs */
generate po = princ / 1000000
generate lpo = ln(po+1)
generate ptpno = (lpo>0)
replace spr = spr*100

foreach t of varlist td bis {
	foreach i of varlist cg-no {
		generate `i'_`t' = `i'*`t'
	}
}

foreach t of varlist td bis {
	generate `t'_bnpl = `t'*bnpl
	foreach v of varlist bsgs-bcgln {
		generate `t'_`v' = `t'*`v'
		generate `t'_`v'_bnpl = `t'*`v'*bnpl
	}
}

foreach t of varlist td bis {
	generate `t'_lnpl = `t'*lnpl
	foreach v of varlist lsgs-lcgln {
		generate `t'_`v' = `t'*`v'
		generate `t'_`v'_lnpl = `t'*`v'*lnpl
	}
}


//  TABLE 7 - BORROWER REACTIONS BY COLLATERAL TYPE
quietly reghdfe lpo bis_bnpl bis_bcgln bis_bcgln_bnpl if cg == 1 & lbcpos > 0, absorb (br ln#day) vce(cluster br ln day)
estimates store m1, title("cgs")
quietly reghdfe ptpno bis_bnpl bis_bcgln bis_bcgln_bnpl if cg == 1 & lbcpos > 0, absorb (br ln#day) vce(cluster br ln day)
estimates store m2, title("cgs")
quietly reghdfe lpo bis_bnpl bis_bsgln bis_bsgln_bnpl if sg == 1 & lbcpos > 0, absorb (br ln#day) vce(cluster br ln day)
estimates store m3, title("sgs")
quietly reghdfe ptpno bis_bnpl bis_bsgln bis_bsgln_bnpl if sg == 1 & lbcpos > 0, absorb (br ln#day) vce(cluster br ln day)
estimates store m4, title("sgs")
quietly reghdfe lpo bis_bnpl bis_bcgln bis_bcgln_bnpl bis_bsgln bis_bsgln_bnpl if cg == 1 & lbcpos > 0, absorb (br ln#day) vce(cluster br ln day)
estimates store m5, title("cgs")
quietly reghdfe ptpno bis_bnpl bis_bcgln bis_bcgln_bnpl bis_bsgln bis_bsgln_bnpl if cg == 1 & lbcpos > 0, absorb (br ln#day) vce(cluster br ln day)
estimates store m6, title("cgs")
quietly reghdfe lpo bis_bnpl bis_bcgln bis_bcgln_bnpl bis_bsgln bis_bsgln_bnpl if sg == 1 & lbcpos > 0, absorb (br ln#day) vce(cluster br ln day)
estimates store m7, title("sgs")
quietly reghdfe ptpno bis_bnpl bis_bcgln bis_bcgln_bnpl bis_bsgln bis_bsgln_bnpl if sg == 1 & lbcpos > 0, absorb (br ln#day) vce(cluster br ln day)
estimates store m8, title("sgs")
esttab m1 m2 m3 m4 m5 m6 m7 m8 /*using cltbrwtables.tex*/, mtitles ///
coeflabels(bis_bnpl "NPL" bis_bcgln "AGS" bis_bcgln_bnpl "NPL*AGS" ///
bis_bsgln "Semis" bis_bsgln_bnpl "NPL*Semis" bis_bsfln "Collateral" bis_bsfln_bnpl "NPL*Collateral") ///
cells(b(star fmt(3)) se(par fmt(3))) legend starlevels(* 0.1 ** 0.05 *** 0.01) varwidth(20) modelwidth(10) ///
r2 title("LPO on brw char, brw*lnd FE, brw lnd day clustering")


// TABLE B1 - BORROWERS' REPO RATE REACTIONS
quietly reghdfe spr const lpo td if cg == 1, absorb(const) vce(cluster br ln day)
estimates store m1, title("cgs")
quietly reghdfe spr lpo const td if sg == 1, absorb(const) vce(cluster br ln day)
estimates store m2, title("sgs")
quietly reghdfe spr lpo bis_bnpl bis_bcgln bis_bcgln_bnpl if cg == 1, absorb(br ln#day) vce(cluster br ln day)
estimates store m3, title("cgs")
quietly reghdfe spr lpo bis_bnpl bis_bsgln bis_bsgln_bnpl if sg == 1, absorb(br ln#day) vce(cluster br ln day)
estimates store m4, title("sgs")
quietly reghdfe spr lpo bis_bnpl bis_bcgln bis_bcgln_bnpl bis_bsgln bis_bsgln_bnpl if cg == 1, absorb(br ln#day) vce(cluster br ln day)
estimates store m5, title("cgs")
quietly reghdfe spr lpo bis_bnpl bis_bcgln bis_bcgln_bnpl bis_bsgln bis_bsgln_bnpl if sg == 1, absorb(br ln#day) vce(cluster br ln day)
estimates store m6, title("sgs")
esttab m1 m2 m3 m4 m5 m6 /*using sprtables.tex*/, mtitles ///
/* keep(bis bis_bsgln_bnpl bis_bcgln_bnpl bis_bsfln_bnpl) */ ///
cells(b(star fmt(3)) se(par fmt(3))) legend starlevels(* 0.1 ** 0.05 *** 0.01) varwidth(20) modelwidth(10) ///
r2 title("spr on char, brw FE, brw lnd day clustering")

